import asyncio
import socket
import os

from pylog.pylogger import PyLogger

from py_pli.pylib import VUnits
from py_pli.pylib import GlobalVar

from fleming.common.firmware_util import *

from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal
from virtualunits.meas_seq_generator import MeasurementChannel
from virtualunits.meas_seq_generator import IntegratorMode
from virtualunits.meas_seq_generator import AnalogControlMode


pd_channel = {
    'ref' : MeasurementChannel.REF,
    'abs' : MeasurementChannel.ABS,
    'aux' : MeasurementChannel.AUX,
}

pd_sample = {
    'ref' : TriggerSignal.SampleRef,
    'abs' : TriggerSignal.SampleAbs,
    'aux' : TriggerSignal.SampleAux,
}

pd_reset = {
    'ref' : OutputSignal.Deprc_IntRstRef,
    'abs' : OutputSignal.Deprc_IntRstAbs,
    'aux' : OutputSignal.Deprc_IntRstAux,
}

integrator_mode = {
    'reset'     : IntegratorMode.full_reset,
    'low_reset' : IntegratorMode.low_range_reset,
    'auto'      : IntegratorMode.integrate_autorange,
    'fixed'     : IntegratorMode.integrate_with_fixed_range,
    'low'       : IntegratorMode.integrate_in_low_range,
    'high'      : IntegratorMode.integrate_in_high_range,
}

ANALOGCONTROLMODE_ADD_TO_HIGH_OFFSET            = 1
ANALOGCONTROLMODE_ADD_TO_LOW_OFFSET             = 2
ANALOGCONTROLMODE_ADD_TO_LOW_AND_HIGH_OFFSET    = 3


async def pd_set_mode(name, mode):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    if (name not in pd_channel):
        raise ValueError(f"name must be {pd_channel}")
    if (mode not in integrator_mode):
        raise ValueError(f"mode must be {integrator_mode}")
    
    op_id = 'pd_set_mode'
    seq_gen = meas_seq_generator()
    seq_gen.SetIntegratorMode(**{name: integrator_mode[mode]})
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)

    return f"pd_set_mode() done"


async def pd_set_reset(name, enable):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    if (name not in pd_channel):
        raise ValueError(f"name must be {pd_channel}")
        
    op_id = 'pd_set_reset'
    seq_gen = meas_seq_generator()
    if (enable == 1):
        seq_gen.SetSignals(pd_reset[name])
    else:
        seq_gen.ResetSignals(pd_reset[name])
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, 2)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)

    return f"pd_set_reset() done"


async def pd_measure(name, mode, window_us, fixed_range_us=20):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    full_reset_delay = 10000000 # 100 ms (min 400 us)
    low_reset_delay  =     1200 #  12 us (high range offset measurement needs slightly over 11 us)

    if (name not in pd_channel):
        raise ValueError(f"name must be {pd_channel}")
    if (mode not in integrator_mode):
        raise ValueError(f"mode must be {integrator_mode}")
    if (window_us < 12) or (window_us > 671088):
        raise ValueError(f"window_us must be in the range [12, 671088] us")
    if (fixed_range_us < 0) or (fixed_range_us > 671088):
        raise ValueError(f"fixed_range_us must be in the range [0, 671088] us")
        
    window      = round(window_us * 100)
    fixed_range = round(fixed_range_us * 100)
    if (fixed_range >= window):
        fixed_range = 0

    op_id = 'pd_measure'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # high_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # high_result
    # Reset the offsets
    seq_gen.SetAnalogControl(**{name: AnalogControlMode.full_offset_reset})
    # Enable full reset
    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(**{name: IntegratorMode.FULL_RESET})
    # Measure the high range offset, then enable low range reset
    seq_gen.TimerWaitAndRestart(low_reset_delay)
    seq_gen.SetTriggerOutput(pd_sample[name])
    seq_gen.SetIntegratorMode(**{name: IntegratorMode.LOW_RESET})
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=3)
    if (mode != 'high'):
        seq_gen.SetAnalogControl(**{name: AnalogControlMode.read_offset})
    # Start the integrator (250 ns before the measurement window starts -> TBD)
    seq_gen.TimerWaitAndRestart(25)
    seq_gen.SetIntegratorMode(**{name: integrator_mode[mode]})
    # Measure the analog offset, this is the start of the measurement window
    seq_gen.TimerWaitAndRestart(window - fixed_range)
    seq_gen.SetTriggerOutput(pd_sample[name])
    if (fixed_range > 0):
        # Switch to fixed range integration
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetIntegratorMode(**{name: IntegratorMode.integrate_with_fixed_range})
    # Load the measured offset into the register
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=3)
    seq_gen.SetAnalogControl(**{name: AnalogControlMode.read_offset})
    # Measure the result
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(pd_sample[name])
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=1)
    # Enable full reset
    seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})
    # Reset the offsets (only needed in case a sequence without offset correction is called after this)
    seq_gen.SetAnalogControl(**{name: AnalogControlMode.full_offset_reset})
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, 4)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    PyLogger.logger.info(f"Low: {results[0]:5d} ; High: {results[1]:5d} ; (Offset: {results[2]:5d} ; {results[3]:5d})")

    return results


async def pd_measure_v7(name, mode, window_us, fixed_range_us=20, offset=0, wait_for_trigger=0):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    full_reset_delay = 10000000     # 100 ms (min 400 us)
    conversion_delay = 1200         #  12 us
    trigger_delay = 100000 - 1000   # 990 us (1 kHz signal, starting 10 us after window)
    switch_delay = 25               # 250 ns delay to counter the charge injection

    if (name not in pd_channel):
        raise ValueError(f"name must be {pd_channel}")
    if (mode not in integrator_mode):
        raise ValueError(f"mode must be {integrator_mode}")
    if (window_us < 12) or (window_us > 671088):
        raise ValueError(f"window_us must be in the range [12, 671088] us")
    if (fixed_range_us < 0) or (fixed_range_us > 671088):
        raise ValueError(f"fixed_range_us must be in the range [0, 671088] us")
        
    window      = round(window_us * 100)
    fixed_range = round(fixed_range_us * 100)
    if (fixed_range >= window):
        fixed_range = 0

    op_id = 'pd_measure'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # high_offset
    # Reset the offsets
    seq_gen.SetAnalogControl(**{name: AnalogControlMode.full_offset_reset})
    # Set D(0) to the negation of the offset (16 bit two's complement)
    seq_gen.LoadDataReg(stackNotRegDst=False, dstReg=0, value=(-offset & 0xFFFF))
    # Enable full reset
    seq_gen.TimerWaitAndRestart(full_reset_delay - conversion_delay)
    seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})
    # Measure the high range offset
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(pd_sample[name])
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=3)
    seq_gen.SetAnalogControl(**{name: AnalogControlMode.read_offset})
    # Wait for tigger
    if (wait_for_trigger == 1):
        seq_gen.TimerWait()
        seq_gen.WaitForTriggerInput(TRF=True)
        seq_gen.TimerWaitAndRestart(trigger_delay - switch_delay)
    # Start the integrator in low range (250 ns before the measurement window starts -> TBD)
    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(**{name: IntegratorMode.low_range_reset})
    # Measure the low range offset, this is the start of the measurement window
    seq_gen.TimerWaitAndRestart(window - fixed_range)
    seq_gen.SetTriggerOutput(pd_sample[name])
    seq_gen.SetIntegratorMode(**{name: integrator_mode[mode]})
    if (fixed_range > 0):
        # Switch to fixed range integration
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetIntegratorMode(**{name: IntegratorMode.integrate_with_fixed_range})
    # Load the measured offset into the register
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=3)
    seq_gen.SetAnalogControl(**{name: AnalogControlMode.read_offset})
    # Add negation of offset to both offset registers
    seq_gen.SetAnalogControl(**{name: ANALOGCONTROLMODE_ADD_TO_LOW_AND_HIGH_OFFSET})
    # Measure the result
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(pd_sample[name])
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=1)
    # Enable full reset
    seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})
    # Reset the offsets (only needed in case a sequence without offset correction is called after this)
    seq_gen.SetAnalogControl(**{name: AnalogControlMode.full_offset_reset})
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, 4)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    log_test_info(f"Low: {results[0]:5d} ; High: {results[1]:5d} ; (Offset: {results[2]:5d} ; {results[3]:5d})")

    return results


async def pd_window_scan(name, mode, start_us, stop_us, step_us):
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    with open(f"pd_window_scan_{timestamp}.csv", 'w') as file:
        file.write(f"pd_window_scan(name={name}, mode={mode}, start_us={start_us}, stop_us={stop_us}, step_us={step_us})\n")
        file.write(f"time [ms]   ; analog_low  ; analog_high ; offset_low  ; offset_high\n")
        window_range = [window / 1e6 for window in range(round(start_us * 1e6), round(stop_us * 1e6 + 1), round(step_us * 1e6))]
        for window in window_range:
            results = await pd_measure(name, mode, window, fixed_range_us=0)
            file.write(f"{(window / 1000):11.6f} ; {results[0]:11d} ; {results[1]:11d} ; {results[2]:11d} ; {results[3]:11d}\n")

    return f"pd_window_scan() done"


async def pd_window_scan_mean(name, mode, start_us, stop_us, step_us, iterations=10):
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    with open(f"pd_window_scan_mean_{timestamp}.csv", 'w') as file:
        file.write(f"pd_window_scan_mean(name={name}, mode={mode}, start_us={start_us}, stop_us={stop_us}, step_us={step_us}, iterations={iterations})\n")
        file.write(f"time [ms]   ; analog_low  ; analog_high ; offset_low  ; offset_high\n")
        window_range = [window / 1e6 for window in range(round(start_us * 1e6), round(stop_us * 1e6 + 1), round(step_us * 1e6))]
        for window in window_range:
            mean_results = [0] * 4
            for i in range(iterations):
                results = await pd_measure(name, mode, window, fixed_range_us=0)
                mean_results = [a + b for a, b in zip(mean_results, results)]
            
            mean_results = [sum / iterations for sum in mean_results]
            file.write(f"{(window / 1000):11.6f} ; {mean_results[0]:11.3f} ; {mean_results[1]:11.3f} ; {mean_results[2]:11.3f} ; {mean_results[3]:11.3f}\n")

    return f"pd_window_scan_mean() done"


async def pd_reset_test(name, mode, window_us):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    full_reset_delay = 10000000 # 100 ms (min 400 us)

    if (name not in pd_channel):
        raise ValueError(f"name must be {pd_channel}")
    if (mode not in integrator_mode):
        raise ValueError(f"mode must be {integrator_mode}")
    if (window_us < 12) or (window_us > 671088):
        raise ValueError(f"window_us must be in the range [12, 671088] us")

    window = round(window_us * 100)

    op_id = 'pd_reset_test'
    seq_gen = meas_seq_generator()
    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(**{name: IntegratorMode.FULL_RESET})
    seq_gen.TimerWaitAndRestart(window)
    seq_gen.SetIntegratorMode(**{name: integrator_mode[mode]})
    seq_gen.TimerWait()
    if (mode == 'low'):
        seq_gen.SetIntegratorMode(**{name: IntegratorMode.LOW_RESET})
    else:
        seq_gen.SetIntegratorMode(**{name: IntegratorMode.FULL_RESET})
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)

    return f"pd_reset_test() done"


async def pd_reset_scan(name, mode, window_us, sample_rate=83333, sample_count=500):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    full_reset_delay = 10000000     # 100 ms

    if (name not in pd_channel):
        raise ValueError(f"name must be {pd_channel}")
    if (mode not in integrator_mode):
        raise ValueError(f"mode must be {integrator_mode}")
    if (window_us < 12) or (window_us > 671088):
        raise ValueError(f"window_us must be in the range [12, 671088] us")
    if (sample_rate < 2) or (sample_rate > 83333):
        raise ValueError(f"sample_rate must be in the range [2, 83333] Hz")
    if (sample_count < 1) or (sample_count > 500):
        raise ValueError(f"sample_count must be in the range [1, 500]")

    window = round(window_us * 100)

    sample_delay = round(100000000 / sample_rate)   # 10 ns resolution
    result_count = sample_count * 2

    op_id = 'pd_reset_scan'
    seq_gen = meas_seq_generator()
    # Clear the result buffer
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(result_count)
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()
    # Run the dummy measurement
    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})
    seq_gen.TimerWaitAndRestart(window)
    seq_gen.SetIntegratorMode(**{name: integrator_mode[mode]})
    seq_gen.TimerWaitAndRestart(sample_delay)
    if (mode == 'low'):
        seq_gen.SetIntegratorMode(**{name: IntegratorMode.low_range_reset})
    else:
        seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})
    # Measure the reset curve
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(sample_count)
    seq_gen.SetTriggerOutput(pd_sample[name])
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=True, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=True, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=1)
    seq_gen.TimerWaitAndRestart(sample_delay)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=2)
    seq_gen.LoopEnd()
    if (mode == 'low'):
        seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, result_count)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    with open(f"pd_reset_scan_{timestamp}.csv", 'w') as file:
        file.write(f"pd_reset_scan(name={name}, mode={mode}, window_us={window_us}, sample_rate={sample_rate}, sample_count={sample_count})\n")
        file.write(f"time [ms]   ; analog_low  ; analog_high\n")
        time_ms = 0.0
        for i in range(sample_count):
            file.write(f"{time_ms:11.6f} ; {results[i * 2 + 0]:11d} ; {results[i * 2 + 1]:11d}\n")
            time_ms = time_ms + sample_delay / 100000.0

    return f"pd_reset_scan() done"


async def pdd_addr_reg_test(measurement_count):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    full_reset_delay = 40000    # 400 us
    conversion_delay = 1200     #  12 us
    switch_delay = 25           # 250 ns
    window = 10000              # 100 us
    fixed_range = 2000          #  20 us
    loop_delay = 10000000 - full_reset_delay - switch_delay - window

    op_id = 'pdd_addr_reg_test'
    seq_gen = meas_seq_generator()

    # Clear the result buffer
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(measurement_count * 2)
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()

    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)

    seq_gen.Loop(measurement_count)
    # Reset the offsets
    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset)
    # Enable full reset
    seq_gen.TimerWaitAndRestart(full_reset_delay - conversion_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset)
    # Measure the high range offset
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)
    # Start the integrator in low range (250 ns before the measurement window starts -> TBD)
    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.low_range_reset)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset)
    # Measure the low range offset, this is the start of the measurement window
    seq_gen.TimerWaitAndRestart(window - fixed_range)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_autorange)
    # Switch to fixed range integration
    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_with_fixed_range)
    # Load the measured offset into the register
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset)
    # Measure the result
    seq_gen.TimerWaitAndRestart(loop_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)
    seq_gen.GetAnalogResult(pd_channel['ref'], isRelativeAddr=True, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(pd_channel['ref'], isRelativeAddr=True, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=2)
    seq_gen.LoopEnd()

    # Enable full reset
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset)
    # Reset the offsets (only needed in case a sequence without offset correction is called after this)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset)
    seq_gen.Stop(0)

    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, (measurement_count * 2))
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    return results


async def pdd_analog_limit_test(limit=16384, start_us=185, stop_us=195, step_us=0.1):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    window_range = [window / 1e6 for window in range(round(start_us * 1e6), round(stop_us * 1e6 + 1), round(step_us * 1e6))]

    for window_us in window_range:

        full_reset_delay = 40000    # 400 us
        conversion_delay = 1200     #  12 us
        switch_delay = 25           # 250 ns
        fixed_range = 2000          #  20 us

        window = round(window_us * 100)

        op_id = 'pdd_analog_limit_test'
        seq_gen = meas_seq_generator()

        # Clear the result buffer
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)

        # Write analog limit to data register 0
        # seq_gen.currSequence.append(0x0A000000 | (limit & 0xFFFF))
        seq_gen.LoadDataReg(stackNotRegDst=False, dstReg=0, value=limit)

        # Reset the offsets
        seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset)
        # Enable full reset
        seq_gen.TimerWaitAndRestart(full_reset_delay - conversion_delay)
        seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset)
        # Measure the high range offset
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)
        # Start the integrator in low range (250 ns before the measurement window starts -> TBD)
        seq_gen.TimerWaitAndRestart(switch_delay)
        seq_gen.SetIntegratorMode(ref=IntegratorMode.low_range_reset)
        seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset)
        # Measure the low range offset, this is the start of the measurement window
        seq_gen.TimerWaitAndRestart(window - fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)
        seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_autorange)
        # Switch to fixed range integration
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_with_fixed_range)
        # Load the measured offset into the register
        seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset)
        # Measure the result
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)
        seq_gen.TimerWait()
        seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
        seq_gen.SetAddrRegConditionally(pd_channel['ref'], checkCounterValue=False, relative=False, overridePrevValue=False, checkHiRange=True, dataPosition=0, addressPosition=0, addrValue=2)
        seq_gen.SetAddrRegConditionally(pd_channel['ref'], checkCounterValue=False, relative=False, overridePrevValue=False, checkHiRange=False, dataPosition=0, addressPosition=0, addrValue=2)
        seq_gen.GetAnalogResult(pd_channel['ref'], isRelativeAddr=True, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
        seq_gen.GetAnalogResult(pd_channel['ref'], isRelativeAddr=True, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)

        # Enable full reset
        seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset)
        # Reset the offsets (only needed in case a sequence without offset correction is called after this)
        seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset)
        seq_gen.Stop(0)

        meas_unit.ClearOperations()
        meas_unit.resultAddresses[op_id] = range(0, 4)
        await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        PyLogger.logger.info(f"{window_us:.2f}: {results}")

    return f"pdd_analog_limit_test() done"


async def pdd_analog_limit_test_v2(analog_limit=6554, start_us=10, stop_us=100, step_us=1):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    window_range = [window / 1e6 for window in range(round(start_us * 1e6), round(stop_us * 1e6 + 1), round(step_us * 1e6))]

    for windowTime_us in window_range:

        numberOfFlashes = 1
        freqOfFlashes_Hz = 100      # 100 Hz
        delay_us = 100              # 100 us
        flash_arm_time_us = 500     # 500 us
        conversion_delay_us = 12    #  12 us
        switch_delay = 25           # 250 ns
        reference_window_us = 30    #  30 us
        fixed_range_us = 20         #  20 us

        pre_cnt_window = 100  # 1 µs

        windowTime_us = round(windowTime_us)

        op_id = 'pdd_analog_limit_test'
        seq_gen = meas_seq_generator()

        seq_gen.ClearResultBuffer(relative = False, dword = True, addrReg= 0, addr = 0)
        seq_gen.ClearResultBuffer(relative = False, dword = True, addrReg= 0, addr = 2)
        seq_gen.ClearResultBuffer(relative = False, dword = True, addrReg= 0, addr = 4)
        seq_gen.ClearResultBuffer(relative = False, dword = True, addrReg= 0, addr = 6)
        seq_gen.ClearResultBuffer(relative = False, dword = True, addrReg= 0, addr = 8)
        seq_gen.ClearResultBuffer(relative = False, dword = True, addrReg= 0, addr = 10)
        seq_gen.ClearResultBuffer(relative = False, dword = True, addrReg= 0, addr = 12)
        seq_gen.ClearResultBuffer(relative = False, dword = True, addrReg= 0, addr = 14)
        seq_gen.ClearResultBuffer(relative = False, dword = True, addrReg= 0, addr = 16)

        seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2 | OutputSignal.HVGatePMT2 | OutputSignal.HVGatePMT1)
        #seq_gen.SetDataReg(stackNotRegSrc= False, add= True, sign= True, stackNotRegDst= False, srcReg= 8, dstReg= 8, value= pulse_counter_limit & 0xFFFF)
        #seq_gen.SetDataReg(stackNotRegSrc= False, add= True, sign= True, stackNotRegDst= False, srcReg= 9, dstReg= 9, value= pulse_counter_limit >> 16)
        seq_gen.LoadDataReg(stackNotRegDst= False, dstReg= 10, value= analog_limit)

        seq_gen.SetAddrReg(relative= False, dataNotAddrSrc= False, sign= False, stackNotRegSrc= False, srcReg= 0, dstReg= 0, addr= 0)

        #Should you do this only once??? Outside loop
        seq_gen.SetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

        ### Flash loop
        seq_gen.Loop(numberOfFlashes)

        seq_gen.SetAnalogControl(ref = AnalogControlMode.full_offset_reset, pmt1 = AnalogControlMode.full_offset_reset,
                              abs = AnalogControlMode.full_offset_reset)

        seq_gen.TimerWaitAndRestart(flash_arm_time_us * 100)
        seq_gen.SetSignals(OutputSignal.Flash)

        seq_gen.SetIntegratorMode(ref = IntegratorMode.full_reset, pmt1 = IntegratorMode.full_reset,
                               abs = IntegratorMode.full_reset)
        seq_gen.ResetSignals(OutputSignal.InputGatePMT2 | OutputSignal.InputGatePMT1)
        # Measure the high range offset
        seq_gen.TimerWaitAndRestart(conversion_delay_us * 100)
        seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1)

        # No need to readout offset values
        #seq_gen.GetAnalogResult(pmt_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False,
        #                        addResult=False, dword=False, addrPos=0, resultPos=3)
        #seq_gen.GetAnalogResult(pmt_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True,
        #                        addResult=False, dword=False, addrPos=0, resultPos=4)
        seq_gen.TimerWaitAndRestart(switch_delay)

        # Start the integrator in low range (250 ns before the low range offset is measured -> TBD)
        seq_gen.SetIntegratorMode(ref = IntegratorMode.low_range_reset, pmt1 = IntegratorMode.low_range_reset,
                               abs = IntegratorMode.low_range_reset)
        seq_gen.SetAnalogControl(ref = AnalogControlMode.read_offset, pmt1 = AnalogControlMode.read_offset,
                              abs = AnalogControlMode.read_offset)

        # Measure the low range offset, then switch to auto range
        seq_gen.TimerWaitAndRestart(reference_window_us * 100)
        seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SamplePMT1 | TriggerSignal.SampleAbs)
        seq_gen.SetIntegratorMode(ref = IntegratorMode.integrate_autorange, pmt1 = IntegratorMode.integrate_autorange,
                               abs = IntegratorMode.integrate_autorange)
        seq_gen.ResetSignals(OutputSignal.Flash)
        seq_gen.TimerWaitAndRestart(fixed_range_us * 100)
        seq_gen.SetIntegratorMode(ref = IntegratorMode.integrate_with_fixed_range)

        seq_gen.SetAnalogControl(ref = AnalogControlMode.read_offset)

        seq_gen.TimerWaitAndRestart((delay_us - fixed_range_us - reference_window_us) * 100)

        seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)

        seq_gen.TimerWaitAndRestart(100)   # Delay for InputGating to "rest" in order to generate no counts to PC

        seq_gen.SetSignals(OutputSignal.InputGatePMT2 | OutputSignal.InputGatePMT1)

        seq_gen.TimerWaitAndRestart(pre_cnt_window)  # Start of counting

        seq_gen.PulseCounterControl(channel= 0, cumulative= False, resetCounter= True, resetPresetCounter= True, correctionOn= False)
        seq_gen.PulseCounterControl(channel= 1, cumulative= False, resetCounter= True, resetPresetCounter= True, correctionOn= False)

        seq_gen.Loop(windowTime_us)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel= 0, cumulative= True, resetCounter= False, resetPresetCounter= True, correctionOn= True)
        seq_gen.PulseCounterControl(channel= 1, cumulative= True, resetCounter= False, resetPresetCounter= True, correctionOn= True)
        seq_gen.LoopEnd()

        seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
        seq_gen.TimerWaitAndRestart(fixed_range_us * 100)
        seq_gen.SetIntegratorMode(pmt1 = IntegratorMode.integrate_with_fixed_range,
                               abs = IntegratorMode.integrate_with_fixed_range)

        seq_gen.SetAnalogControl(pmt1 = AnalogControlMode.read_offset,
                              abs = AnalogControlMode.read_offset)

        seq_gen.TimerWaitAndRestart(conversion_delay_us * 100)

        seq_gen.SetTriggerOutput(TriggerSignal.SampleAbs | TriggerSignal.SamplePMT1)

        totalFlashTime = int((1.0 / freqOfFlashes_Hz) * 100000000.0)

        remainingTime = totalFlashTime - ((flash_arm_time_us - windowTime_us - (2 * conversion_delay_us) - delay_us - fixed_range_us) * 100)
        if remainingTime < 0:
            ### Todo: Check parameters before
            raise Exception("TRF Parameter Error")
        seq_gen.TimerWaitAndRestart(remainingTime)

        seq_gen.SetAddrReg(relative= False, dataNotAddrSrc= False, sign= False, stackNotRegSrc= False, srcReg= 0, dstReg= 8, addr= 0)
        seq_gen.SetAddrReg(relative= False, dataNotAddrSrc= False, sign= False, stackNotRegSrc= False, srcReg= 0, dstReg= 9, addr= 0)
        seq_gen.SetAddrRegConditionally(channel= MeasurementChannel.PMT1, checkCounterValue= False, relative= True, overridePrevValue= False, checkHiRange= True, dataPosition= 0, addressPosition= 8, addrValue= 10)
        seq_gen.SetAddrRegConditionally(channel= MeasurementChannel.PMT1, checkCounterValue= False, relative= True, overridePrevValue= False, checkHiRange= False, dataPosition= 10, addressPosition= 8, addrValue= 10)
        #seq_gen.SetAddrRegConditionally(channel= MeasurementChannel.PMT1, checkCounterValue= True, relative= True, overridePrevValue= False, checkHiRange= False, dataPosition= 8, addressPosition= 8, addrValue= 10)
        seq_gen.SetAddrRegConditionally(channel= MeasurementChannel.ABS, checkCounterValue= False, relative= True, overridePrevValue= False, checkHiRange= True, dataPosition= 0, addressPosition= 9, addrValue= 10)
        seq_gen.SetAddrRegConditionally(channel= MeasurementChannel.ABS, checkCounterValue= False, relative= True, overridePrevValue= False, checkHiRange= False, dataPosition= 10, addressPosition= 9 , addrValue= 10)
        #seq_gen.SetAddrRegConditionally(channel= MeasurementChannel.PMT2, checkCounterValue= True, relative= True, overridePrevValue= False, checkHiRange= False, dataPosition= 8, addressPosition= 9, addrValue= 10)

        seq_gen.GetAnalogResult(channel= MeasurementChannel.PMT1, isRelativeAddr= True, ignoreRange= False, isHiRange= False, addResult= True, dword= False, addrPos= 8, resultPos= 0)
        seq_gen.GetAnalogResult(channel= MeasurementChannel.PMT1, isRelativeAddr= True, ignoreRange= False, isHiRange= True, addResult= True, dword= False, addrPos= 8, resultPos= 1)

        seq_gen.GetAnalogResult(channel= MeasurementChannel.ABS, isRelativeAddr= True, ignoreRange= False, isHiRange= False, addResult= True, dword= False, addrPos= 9, resultPos= 2)
        seq_gen.GetAnalogResult(channel= MeasurementChannel.ABS, isRelativeAddr= True, ignoreRange= False, isHiRange= True, addResult= True, dword= False, addrPos= 9, resultPos= 3)

        seq_gen.GetPulseCounterResult(channel= MeasurementChannel.PMT1, relative= True, resetCounter= True, cumulative= True, dword= True, addrPos= 8, resultPos= 4)
        seq_gen.GetPulseCounterResult(channel= MeasurementChannel.PMT2, relative= True, resetCounter= True, cumulative= True, dword= True, addrPos= 9, resultPos= 6)

        seq_gen.GetAnalogResult(channel= MeasurementChannel.REF, isRelativeAddr= False, ignoreRange= False, isHiRange= False, addResult= True, dword= False, addrPos= 0, resultPos= 8)
        seq_gen.GetAnalogResult(channel= MeasurementChannel.REF, isRelativeAddr= False, ignoreRange= False, isHiRange= True, addResult= True, dword= False, addrPos= 0, resultPos= 9)

        seq_gen.LoopEnd()

        #Should you do this only once??? Outside loop
        seq_gen.ResetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

        seq_gen.Stop(0)

        meas_unit.ClearOperations()
        meas_unit.resultAddresses[op_id] = range(0, 18)
        await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        PyLogger.logger.info(f"{windowTime_us:.2f}: {results}")

    return f"pdd_analog_limit_test() done"


async def pd_noise_test(name, sample_rate=83333, sample_count=4096):
    if (name not in pd_channel):
        raise ValueError(f"name must be {pd_channel}")
    if (sample_rate < 2) or (sample_rate > 83333):
        raise ValueError(f"sample_rate must be in the range [2, 83333] Hz")
    if (sample_count < 1) or (sample_count > 4096):
        raise ValueError(f"sample_count must be in the range [1, 4096]")

    await send_gc_msg(f"Start EEF firmware")
    await start_firmware('eef')

    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/pdd_test_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/pd_noise_test__{instrument}_{timestamp}.csv", 'w') as file:
        await write_gc_msg(file, f"pd_noise_test(name={name}, sample_rate={sample_rate}, sample_count={sample_count}) on {instrument} started at {timestamp}")
        await write_gc_msg(file, f"time [ms] ; full_reset ; low_range  ; high_range")

        full_reset_noise = await pd_noise_measurement(name=name, mode='reset', sample_rate=sample_rate, sample_count=sample_count)
        low_range_noise  = await pd_noise_measurement(name=name, mode='low',   sample_rate=sample_rate, sample_count=sample_count)
        high_range_noise = await pd_noise_measurement(name=name, mode='high',  sample_rate=sample_rate, sample_count=sample_count)

        time_ms = 0.0
        for i in range(sample_count):
            await write_gc_msg(file, f"{time_ms:9.3f} ; {full_reset_noise[i]:10d} ; {low_range_noise[i]:10d} ; {high_range_noise[i]:10d}")
            time_ms += 1000.0 / sample_rate

    return f"pd_noise_test() done"


async def pd_noise_measurement(name, mode, sample_rate=83333, sample_count=1000):
    if (name not in pd_channel):
        raise ValueError(f"name must be {pd_channel}")
    if (mode not in integrator_mode):
        raise ValueError(f"mode must be {integrator_mode}")
    if (sample_rate < 2) or (sample_rate > 83333):
        raise ValueError(f"sample_rate must be in the range [2, 83333] Hz")
    if (sample_count < 1) or (sample_count > 4096):
        raise ValueError(f"sample_count must be in the range [1, 4096]")
        
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    full_reset_delay = 40000    # 400 us
    sample_delay = round(100000000 / sample_rate)   # 10 ns resolution

    op_id = 'pd_noise_measurement'
    seq_gen = meas_seq_generator()
    # Clear the result buffer
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(sample_count)
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    # Enable full reset
    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})
    # Sample the signal
    seq_gen.TimerWaitAndRestart(sample_delay)
    seq_gen.SetIntegratorMode(**{name: integrator_mode[mode]})
    seq_gen.Loop(sample_count)
    seq_gen.TimerWaitAndRestart(sample_delay)
    seq_gen.SetTriggerOutput(pd_sample[name])
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=True, ignoreRange=True, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()
    # End of measurement
    seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})
    seq_gen.Stop(0)

    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, sample_count)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    return results


async def pdd_ref_signal(led_current, window_ms, iterations, delay=0):

    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    await send_gc_msg(f"Starting Firmware", log=True)
    await asyncio.gather(
        start_firmware('fmb'),
        start_firmware('eef'),
    )

    await set_led_current(led_current, 'fmb', 'g3', 'green')

    for i in range(iterations):
        if GlobalVar.get_stop_gc():
            return f"pdd_ref_signal stopped by user"

        results = await pd_measure_v7('ref', mode='auto', window_us=(window_ms * 1000))

        await send_gc_msg(f"Low: {results[0]:5d} ; High: {results[1]:5d} ; (Offset: {results[2]:5d} ; {results[3]:5d})", log=False)

        await asyncio.sleep(delay)

    return f"pdd_ref_signal done"


async def set_led_current(current, source='smu', channel='led1', led_type='red'):
    if source.startswith('fmb'):
        return await fmb_set_led_current(current, source, channel, led_type)
    
    raise Exception(f"Invalid source: {source}")


fmb_endpoint = get_node_endpoint('fmb')

fmb_led_channel = {
    'g3' : {'bc':FMBAnalogOutput.BCG, 'gs':FMBAnalogOutput.GSG3},
}

fmb_full_scale_current = {
    'fmb'    :   127,   # Brightness Value [0, 127]
    'fmb#7'  :  2129,   # Measured with Keithley Multimeter (FMB#7,  24k3, Rev 2)
    'fmb#12' :  2084,   # Measured with Keithley Multimeter (FMB#12, 24k3, Rev 2)
    'fmb#15' : 11084,   # Measured with Keithley Multimeter (FMB#15,  4k7, Rev 2)
}

async def fmb_set_led_current(current=17, source='fmb', channel='led1', led_type='green'):
    bc = fmb_led_channel[channel]['bc']
    gs = fmb_led_channel[channel]['gs']
    full_scale_current = fmb_full_scale_current[source]
    brightness = current / full_scale_current
    current = round(min(int(brightness * 128), 127) / 127 * full_scale_current)
    if (current > 0):
        await fmb_endpoint.SetAnalogOutput(bc, brightness)
        await fmb_endpoint.SetAnalogOutput(gs, 1.0)
    else:
        await fmb_endpoint.SetAnalogOutput(bc, 0.0)
        await fmb_endpoint.SetAnalogOutput(gs, 0.0)

    await asyncio.sleep(0.2)
    return current

